<?php
/**
 * Opakowanie Memcache
 */
/**
 * Opakowanie Memcache
 *
 * Umożliwia podział bufora na partycje,
 * które można niezależnie czyścić.
 * 
 * Wykorzystuje klucze skłądające się z nazwy partycji, aktualnego
 * klucza przestrzeni nazw i klucza zapisanych elementów,
 * np. sql_128_$sha1ofquery
 */
class Cache_Memcache {
  /**
   * @var bool Czy mamy połączenie przynajmniej z jednym serwerem w puli
   */
  protected $connected = false;
  /**
   * @var Memcache
   */
  protected $memcache = null;
  protected $pool = array(
      array('host' => 'localhost', 'port' => '11211', 'weight' => 1),
          // Tu powinny być definicje innych hostów
  );
  /**
   * Konstruktor
   */
  public function __construct() {
    $this->connect();
  }
  public function isConnected() {
    return $this->connected;
  }
  /**
   * Łączy z pulą memcached
   *
   * @return void
   */
  protected function connect() {
    $this->connected = false;
    $this->memcache = new Memcache();
    foreach ($this->pool as $host) {
      $this->memcache->addServer($host['host'], $host['port'], true, $host['weight']);
      // Potwierdzenie połączenia przynajmniej z jednym serwerem z puli
      $stats = $this->memcache->getExtendedStats();
      if ($this->connected || ($stats["{$host['host']}:{$host['port']}"] !== false && sizeof($stats["{$host['host']}:{$host['port']}"]) > 0)) {
        $this->connected = true;
      }
    }
    return $this->connected;
  }
  /**
   * Zwraca wartość przestrzeni nazw dla bieżącej partycji
   * 
   * Ta metoda tworzy nowy klucz przestrzeni nazw dla bieżącej partycji.
   * 
   * Aby wyczyścić bufor określonej partycji, należy tylko zwiększyć
   * ten klucz.
   *
   * @param string $key
   * @return string
   */
  protected function addNamespace($partition = '') {
    // Jeśli nie ma połączenia, zwraca false
    if (!$this->connected) {
      return false;
    }
    // Pobranie aktualnego klucza przestrzeni nazw
    $ns_key = $this->memcache->get($partition);
    if ($ns_key == false) {
      // Aktualnie nie ma ustawionego klucza, a więc ustawiamy losowy
      $ns_key = rand(1, 10000);
      $result = $this->memcache->set($partition, $ns_key, 0, 0);
    }
    // Zwrot klucza z kluczem przestrzeni nazw
    $my_key = $partition . "_" . $ns_key . "_" . $key;
    return $my_key;
  }
  /**
   * Czyści bufor poprzez zwiększenie klucza przestrzeni nazw
   *
   * @return void
   */
  public function clearCache($partition = '') {
    if (!$this->connected) {
      return false;
    }
    // Memcache ma wbudowaną metodę inkrementacji
    $this->memcache->increment($partition);
  }
  /**
   * Dodaje wartość do bufora
   * 
   * Będziemy także dodawać klucz metadanych
   * z datą modyfikacji oraz automatycznie 
   * dzielić duże wartości (>=1MB) na
   * kilka kluczy.
   *
   * @param string $key
   * @param string $value
   * @param int $expires
   * @return boolean
   */
  public function set($key, $value, $partition = '', $expires = 14400) {
    // Definicja stałej, aby nie było w programie magicznej liczby
    define('ONE_MB', 1 * 1024 * 1024);
    if (!$this->connected) {
      return false;
    } elseif (strlen($value) >= ONE_MB) {
      // Wartość ma rozmiar powyżej 1 MB, a kwalifikuje się do podziału
      $value = str_split($value, ONE_MB);
    }
    // Ustawienie czasu wygaśnięcia
    if ($expires !== 0) {
      $expires += time();
    }
    // Dodanie partycji i klucza przestrzeni nazw do klucza elementu
    $ns_key = $this->addNameSpace($key, $partition);
    $this->memcache->set($ns_key . '_metadata', json_encode((object) array("modified" => gmdate('D, d M Y H:i:s') . ' GMT', 'slabs' => sizeof($value))),MEMCACHE_COMPRESSED, $expires);
    // Jeśli wartość jest podzielona, musimy ją zapisać w kilku kluczach
    if (is_array($value)) {
      foreach ($value as $k => $v) {
        // Dodanie zwiększonej liczby do klucza i zapisanie fragmentu
        $this->memcache->set($ns_key . '_' . $k, $v, MEMCACHE_COMPRESSED, $expires);
      }
      return true;
    }
    return $this->memcache->set($ns_key, $value, MEMCACHE_COMPRESSED, $expires);
  }
  /**
   * Zwraca dane odpowiadające kluczowi. 
   * 
   * Zwraca false, jeśli nie ma danych.
   *
   * Automatycznie pobiera klucz metadanych
   * i wysyła nagłówek Last-Modified.
   * 
   * Automatycznie pobiera duże wartości podzielone
   * na kilka części.
   * 
   * Dodatkowo wysyła nagłówek X-Cache-Hit informujący, czy
   * element został znaleziony w buforze.
   *
   * @param string $key
   * @return string
   */
  public function get($key, $partition = '') {
    if (!$this->connected) {
      return false;
    }
    $ns_key = $this->addNameSpace($key, $partition);
    $meta = $this->memcache->get($ns_key . '_metadata');
    // Wysłanie odpowiednich nagłówków
    if ($meta && !empty($meta) && !headers_sent()) {
      $meta = json_decode($meta);
      header("X-Cache-Hit: 1", false);
      if (isset($meta->modified)) {
        header('Last-Modified: ' . $meta->modified);
      }
    } elseif (!$meta && !headers_sent()) {
      header("X-Cache-Hit: 0", false);
      return false;
    }
    // Pobranie danych podzielonych na części
    $value = '';
    if ($meta && isset($meta->slabs) && $meta->slabs > 1) {
      // Element jest podzielony
      for ($i = 0; $i < $meta->slabs; $i++) {
        // Dołączenie każdego klucza do wcześniej zwróconych danych
        $value .= $this->memcache->get($ns_key . '_' . $i);
      }
    } else {
      // Element nie jest podzielony
      $value = $this->memcache->get($ns_key);
    }
    return $value;
  }
  /**
   * Usuwa dane odpowiadające określonemu kluczowi.
   * 
   * Zwraca true w razie powodzenia i false w razie niepowodzenia.
   *
   * @param string $key
   * @return boolean
   */
  public function delete($key, $partition = '') {
    if (!$this->connected) {
      return false;
    }
    return $this->memcache->delete($this->addNamespace($key, $partition));
  }
}

?>